import os
import sys
import subprocess
import platform
import argparse


def fast_tests(component_name_input: str = None, no_migrate: bool = False, coverage: bool = False):
    """
    Запускає pytest для компоненти або конкретного файлу тесту в компоненті.
    Підтримує вкладені директорії тестів та аналіз покриття.

    Args:
        component_name_input: Назва компоненти або 'компонента [директорія] файл_тесту.py'
        no_migrate: Якщо True, пропускає міграції
        coverage: Якщо True, вмикає аналіз покриття коду
    """
    # Визначаємо абсолютні шляхи
    k2_root = os.path.abspath(os.path.dirname(__file__))
    venv_path = os.path.dirname(k2_root)

    # Визначаємо Python з venv
    if platform.system() == 'Windows':
        python_executable = os.path.join(venv_path, "venv", "Scripts", "python.exe")
    else:
        python_executable = os.path.join(venv_path, "venv", "bin", "python")

    if not os.path.isfile(python_executable):
        print(f"❌ Не знайдено Python: {python_executable}")
        return False

    if not component_name_input:
        print("❌ Назва компоненти не вказана.")
        return False

    # Розбір вводу
    parts = component_name_input.split()
    component_part = parts[0]
    test_path_parts = parts[1:]

    # --- ЗМІНЕНО: Спрощена та надійніша логіка визначення файлу/директорії ---
    test_file_part = None
    dir_parts = []
    if test_path_parts:
        last_part = test_path_parts[-1]
        if last_part.endswith('.py'):
            test_file_part = last_part
            dir_parts = test_path_parts[:-1]
        else:
            # Всі частини шляху є директоріями
            dir_parts = test_path_parts

    if test_file_part and not test_file_part.lower().endswith('.py'):
        test_file_part += '.py'

    component_part_norm = os.path.normpath(component_part).replace(os.path.sep, "_")

    # Визначаємо базовий шлях тестів
    component_test_path = os.path.join("components", component_part_norm, component_part_norm, "tests")
    abs_component_test_path = os.path.join(k2_root, component_test_path)

    if not os.path.isdir(abs_component_test_path):
        print(f"❌ Директорію тестів не знайдено: {abs_component_test_path}")
        return False

    # --- ДОДАНО: Виправлення для уникнення дублювання назви компоненти у шляху ---
    if dir_parts and dir_parts[0] == component_part:
        print(f"ℹ️  Надлишкова назва компоненти '{dir_parts[0]}' у шляху видалена.")
        dir_parts.pop(0)
    # -------------------------------------------------------------------------

    # Готуємо середовище
    env = os.environ.copy()
    python_path = [
        k2_root,
        os.path.join(k2_root, "components"),
        os.path.join(k2_root, "components", component_part_norm),
        venv_path
    ]
    env['PYTHONPATH'] = os.pathsep.join(python_path + sys.path)
    env['FLASK_APP'] = 'k2.app'
    env['TESTING'] = '1'

    if no_migrate:
        env['NO_MIGRATE'] = '1'

    # Формуємо команду pytest
    command = [
        python_executable, "-m", "pytest",
        "--rootdir", k2_root,
        "-v", "-s", "-rs",
        "--tb=long",
        "--disable-warnings", "-p", "no:warnings",
        "--confcutdir", k2_root
    ]

    # Додаємо логіку для аналізу покриття
    if coverage:
        cov_path = os.path.join("components", component_part_norm, component_part_norm).replace("\\", "/")
        print(f"📊 Увімкнено аналіз покриття для: {cov_path}")
        command.extend([
            f"--cov={cov_path}",
            "--cov-report=term-missing",
            "--cov-report=html",
        ])

    # Формуємо цільовий шлях для тестів
    if dir_parts or test_file_part:
        target_path_parts = [component_test_path] + dir_parts
        if test_file_part:
            target_path_parts.append(test_file_part)
        test_path = os.path.join(*target_path_parts)
        command.append(test_path)
        print(f"\nЗапуск тестів для '{test_path}'...")
    else:
        command.append(component_test_path)
        print(f"\nЗапуск всіх тестів для '{component_part}'...")

    print(f"\n🚀 Команда: {' '.join(command)}")
    print(f"📌 Корінь проекту: {k2_root}")
    print(f"🔧 PYTHONPATH: {env['PYTHONPATH']}")
    if no_migrate:
        print(f"⚠️  Міграції вимкнено (NO_MIGRATE=1)")
    print("-" * 50)

    try:
        process = subprocess.Popen(
            command, cwd=k2_root, env=env,
            stdout=subprocess.PIPE, stderr=subprocess.STDOUT,
            bufsize=1, universal_newlines=True, encoding='utf-8',
        )
        for line in process.stdout:
            print(line, end='')
        process.wait()

        if process.returncode == 0:
            print("\n" + "=" * 50)
            print("✅ Тести пройшли успішно.")
            return True
        else:
            print("\n" + "=" * 50)
            print(f"❌ Тести завершились з помилками (код {process.returncode})")
            return False
    except Exception as e:
        print("\n" + "=" * 50)
        print(f"❌ Невідома помилка при запуску тестів: {e}")
        return False

def parse_args():
    """Парсинг аргументів командного рядка або запит до користувача."""
    parser = argparse.ArgumentParser(
        description='Запуск тестів для компоненти',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Приклади використання:
  python k2fast_test.py
  python k2fast_test.py --coverage                      
  python k2fast_test.py k2production                     
  python k2fast_test.py k2production --coverage          
  python k2fast_test.py k2production tests_logbook test_update_spare_parts.py no_migrate
""")
    parser.add_argument('args', nargs='*', help='Компонента, шлях до тесту, та/або no_migrate')
    parser.add_argument('--coverage', action='store_true', help='Увімкнути аналіз покриття коду тестами')
    args = parser.parse_args()

    input_parts = args.args
    coverage = args.coverage

    if not input_parts and not coverage:
        user_input = input("\nВведіть назву компоненти та шлях до тесту [або прапори]: ").strip()
        if not user_input:
            return None, False, False
        input_parts = user_input.split()

    component_input = []
    no_migrate = False

    for part in input_parts:
        if part == 'no_migrate':
            no_migrate = True
        elif part == '--coverage':
            coverage = True
        else:
            component_input.append(part)

    component_name_input = ' '.join(component_input) if component_input else None

    if not component_name_input and (coverage or no_migrate):
        prompt = "Введіть назву компоненти"
        if coverage:
            prompt += " для аналізу покриття"
        component_name_input = input(f"\n{prompt}: ").strip()
        if not component_name_input:
            return None, False, False

    return component_name_input, no_migrate, coverage


if __name__ == "__main__":
    component_input, no_migrate, coverage = parse_args()
    if component_input:
        fast_tests(component_input, no_migrate, coverage)
