import logging
import os
import time
from datetime import datetime
import uuid
import re
import json

from flask_login import LoginManager, current_user
from flask import request, send_from_directory
from sqlalchemy.orm import sessionmaker
from sqlalchemy.orm.attributes import flag_modified


from k2.k2obj import K2Obj


class K2Settings(K2Obj):
    '''клас збереження та пошуку налаштувань  для користувача, складу, контрагента, проекту
    '''
    name = 'k2settings'
    def __init__(self, *args, **kwargs):
        super().__init__()

    #     self._data_path = None
    #
    # @property
    # def data_path(self):
    #     """Getter"""
    #     return self._data_path
    #
    # @data_path.setter
    # def name_yml(self, name: str):
    #     """Setter method"""
    #     self._data_path = name

    def save_settings(self, settings, object, device=None, is_for_all=False, show_message=True):
        '''
        Метод для збереження налаштувань для користувача, складу, контрагента або проєкту.

        Args:
            settings (dict): Словник з налаштуваннями для збереження.
            object (str): Об'єкт, до якого відносяться налаштування (назва гріда, форми тощо).
            device (str, optional): Пристрій, для якого специфічні налаштування. За замовчуванням None.
            is_for_all (bool, optional): Зберегти налаштування для поточного користувача чи для всіх.

        Examples:
            K2().settings.save_settings(settings=json_dict, object='users')
            K2().settings.save_settings(settings=json_dict, object='k2counterparts', is_for_all=True)

        Returns:
            None
        '''
        # Створюємо сесію для роботи з БД
        # Session = sessionmaker(bind=K2.db.engine)
        # session = Session()
        try:
            print(f"-----------------------------[{datetime.now()}] Received save_settings call for object '{object}' with data: {settings}")

            # Імпортуємо необхідні компоненти та моделі
            from k2.k2cfg import K2
            from components.k2site.k2site.models import K2settings
            k2_instance = K2()

            if is_for_all:
                # Логіка для збереження налаштувань "для всіх" (в рамках проєкту/контрагента і т.д.)
                if (k2_instance.secur.is_superadmin(current_user)
                        or k2_instance.secur.is_superadmin_branch(current_user)
                        or k2_instance.secur.is_superadmin_root(current_user)
                        or k2_instance.secur.is_superadmin_cloud(current_user)):

                    roleid = current_user.get_role_id()
                    projid = k2_instance.get_user_project_id()
                    counterpart_id = k2_instance.get_user_counterparts_id()
                    storage_id = k2_instance.get_user_storage_id()

                    # Перевіряємо, чи існують вже такі налаштування
                    existing_setting = K2.db.session.query(K2settings).filter_by(
                        object=object,
                        projid=projid,
                        counterpart_id=counterpart_id,
                        storage_id=storage_id
                    ).first()
                    time.sleep(3)
                    if existing_setting:
                        # Оновлюємо існуючий запис
                        existing_setting.settings = settings
                        # Явно позначаємо поле JSON як змінене
                        flag_modified(existing_setting, "settings")
                        existing_setting.device = device
                        existing_setting.updateuser = current_user.get_id()
                        existing_setting.updatedate = datetime.now()
                    else:
                        # Створюємо новий запис
                        new_setting = K2settings(
                            settingsid=K2.generate_id(),
                            roleid=roleid,
                            device=device,
                            settings=settings,
                            object=object,
                            projid=projid,
                            counterpart_id=counterpart_id,
                            storage_id=storage_id,
                            createuser=current_user.get_id(),
                            createdate=datetime.now(),
                            updateuser=current_user.get_id(),
                            updatedate=datetime.now(),
                            active=1
                        )
                        K2.db.session.add(new_setting)

                    K2.db.session.commit()
                    K2.logging_message(status=K2.log_success, message="Налаштування успішно збережено",
                                       show_message=show_message)
                else:
                    K2.logging_message(status=K2.log_warning, message="Помилка доступу", show_message=show_message)
            else:
                # Логіка для збереження персональних налаштувань користувача
                user_id = current_user.get_id()
                existing_setting = K2.db.session.query(K2settings).filter_by(
                    user_id=user_id,
                    object=object,
                ).first()

                if existing_setting:
                    # Оновлюємо існуючий запис
                    existing_setting.settings = settings
                    # Явно позначаємо поле JSON як змінене
                    flag_modified(existing_setting, "settings")
                    existing_setting.device = device
                    existing_setting.updateuser = current_user.get_id()
                    existing_setting.updatedate = datetime.now()
                    existing_setting.active = 1
                else:
                    # Створюємо новий запис
                    roleid = current_user.get_role_id()
                    new_setting = K2settings(
                        settingsid=K2.generate_id(),
                        roleid=roleid,
                        user_id=user_id,
                        device=device,
                        settings=settings,
                        object=object,
                        createuser=current_user.get_id(),
                        createdate=datetime.now(),
                        updateuser=current_user.get_id(),
                        updatedate=datetime.now(),
                        active=1
                    )
                    K2.db.session.add(new_setting)

                K2.db.session.commit()
                K2.logging_message(status=K2.log_success, message="Налаштування успішно збережено",
                                   show_message=show_message)

        except Exception as e:
            # Для кращої діагностики можна додати повний трейсбек
            import traceback
            detailed_error = traceback.format_exc()
            K2.logging_message(status=K2.log_error, message=f"Помилка в методі save_settings: {e}\n{detailed_error}",
                               show_message=show_message)
            K2.db.session.rollback()  # Важливо відкотити транзакцію в разі помилки
        finally:
            K2.db.session.close()  # Завжди закриваємо сесію


    def del_settings(self, object, device=None):
        '''
        Method to dell settings for a user, storage, counterpart, or project.

        Args:
            settings (dict): A dictionary containing the settings to be saved. This will be converted to a JSON string.
            user_id (str): The ID of the user for whom the settings are being saved.
            object (str): The object for which the settings are specific (name of grid or form or other object).
            device (str, optional): The device for which the settings are specific, if any. Defaults to None.
            is_for_all (Boolean, optional): Save settings for current user or for all.

        Examples:
            K2().settings.save_settings(settings=json_dict,object='users')
            K2().settings.save_settings(settings=json_dict, object='k2counterparts', is_for_all=True)

        Returns:
            None
        '''
        # Session = sessionmaker(bind=K2.db.engine)
        # session = Session()
        try:
            from k2.k2cfg import K2
            from components.k2site.k2site.models import K2settings
            k2_instance = K2()
            user_id = current_user.get_id()
            if (k2_instance.secur.is_superadmin(current_user)
                    or k2_instance.secur.is_superadmin_branch(current_user)
                    or k2_instance.secur.is_superadmin_root(current_user)
                    or k2_instance.secur.is_superadmin_cloud(current_user)):
                # roleid = self.__current_usr_role(current_user.get_id())
                roleid = current_user.get_role_id()
                projid = k2_instance.get_user_project_id()
                counterpart_id = k2_instance.get_user_counterparts_id()
                storage_id = k2_instance.get_user_storage_id()
                # Check if settings already exist
                existing_setting = K2.db.session.query(K2settings).filter_by(
                    object=object, active=1
                    ).all()
                if existing_setting:
                    for existing_setting in existing_setting:
                        # Update existing settings
                        existing_setting.device = device
                        existing_setting.updateuser = current_user.get_id()
                        existing_setting.updatedate = datetime.now()
                        existing_setting.active = 0
            else:
                K2.logging_message(status=K2.log_warning,  message="Permission error", show_message=True)
            # else:
            #     user_id = current_user.get_id()
            #     existing_setting = session.query(K2settings).filter_by(
            #         user_id=user_id,
            #         object=object,
            #     ).first()
            #     if existing_setting:
            #         # Update existing settings
            #         existing_setting.device = device
            #         existing_setting.updateuser = current_user.get_id()
            #         existing_setting.updatedate = datetime.now()
            #         existing_setting.active = 0
            # Створюємо детальний звіт про помилку
            error_message = f"Error in method save_settings: {e}"
            # Виводимо в консоль/лог і те, і інше
            K2.db.session.commit()
            K2.logging_message(status=K2.log_success, message=f"Settings deleted", show_message=True)
        except Exception as e:
            K2.logging_message(status=K2.log_error, message=f"Error in method del_settings {e}", show_message=True)
        # finally:
        #     session.close()

    def _get_setting_record(self, object, user_settings=True):
        '''
        Private helper method to get a setting record for the current user
        according to the hierarchy: user -> storage -> counterpart -> project.
        This method contains the core logic to avoid code duplication.

        Args:
            object (str): The object for which the settings are specific.

        Returns:
            K2settings or None: The found settings record object, otherwise None.
        '''
        # This private method will be used by both get_settings and get_settingsid
        # Session = sessionmaker(bind=K2.db.engine)
        # session = Session()
        try:
            from components.k2site.k2site.models import K2settings, K2users
            from k2.k2cfg import K2

            user_id = current_user.get_id()
            user = K2.db.session.query(K2users).get(user_id)
            if not user:
                return None

            if user_settings:
                # 1. Check for user-specific settings
                setting_record = K2.db.session.query(K2settings).filter_by(user_id=user_id, object=object, active=1).first()
                if setting_record:
                    return setting_record

            # 2. Check for storage-specific settings
            setting_record = K2.db.session.query(K2settings).filter_by(storage_id=user.storage_id, object=object,
                                                                 active=1).first()
            if setting_record:
                return setting_record

            # 3. Check for counterpart-specific settings
            setting_record = K2.db.session.query(K2settings).filter_by(counterpart_id=user.counterpart_id, object=object,
                                                                 active=1).first()
            if setting_record:
                return setting_record

            # 4. Check for project-specific settings
            setting_record = K2.db.session.query(K2settings).filter_by(projid=user.projid, object=object, active=1).first()
            if setting_record:
                return setting_record

            return None
        except Exception as e:
            logging.error(e)
        # finally:
        #     session.close()

    def get_settings(self, object, user_settings=True):
        '''
        Method to get settings for the current user according to the hierarchy.

        Args:
            object (str): The object for which the settings are specific (name of grid or form or other object).

        Returns:
            dict or None: The settings dictionary if found, otherwise None.
        '''
        try:
            setting_record = self._get_setting_record(object, user_settings=user_settings)
            return setting_record.settings if setting_record else None
        except Exception as e:
            logging.error(f"Error in method get_settings: {e}")
            return None

    def get_settingsid(self, object, user_settings=True):
        '''
        Method to get the ID of a settings record for the current user according to the hierarchy.

        Args:
            object (str): The object for which the settings are specific (name of grid or form or other object).

        Returns:
            str or None: The settings ID if found, otherwise None.
        '''
        try:
            setting_record = self._get_setting_record(object, user_settings=user_settings)
            return setting_record.settingsid if setting_record else None
        except Exception as e:
            # Corrected error message
            logging.error(f"Error in method get_settingsid: {e}")
            return None
