# k2cfg.py — оптимізована версія

import os
import sys
import json
import yaml
import pytz
import uuid
import csv
import shutil
import logging
import inspect
import platform
from pathlib import Path
from datetime import datetime, timedelta
from json.decoder import JSONDecodeError
from typing import List, Dict, Any
import time


from flask import (
    g, session, request, jsonify, has_request_context, current_app
)
from flask_login import LoginManager, current_user, logout_user
from sqlalchemy import desc, text, create_engine
from sqlalchemy.orm import joinedload, aliased

from k2.k2obj import K2Obj
from k2.k2secur import K2Secur
from k2.k2path import K2Path
from k2.k2datasync import K2Datasync
from k2.k2settings import K2Settings
from k2.k2notifications import K2Notifications



class K2(K2Obj):
    # --------- базові метадані / конфіг ---------
    name = 'k2cfg'
    version = '2.5.1.79'
    port = ''
    domain_name = ''
    domain = ''
    domain_protocol = 'http://'  # Protocol
    update_domain = ''

    components_install = []
    menu = []
    menu_url = []
    menu_category = []

    sglalchemy_pool_size = 10
    sglalchemy_max_overflow = 20

    safe_mode = False
    json_sort_keys = False
    babel_translation_directories = ''

    default_language = 'uk'
    current_language = 'uk'
    timezone = pytz.timezone('Europe/Kiev')
    secret_key = 'JH&INH987gFDHdsagh&8dwbjdw8ckw'

    generate_migration = ''
    upgrade_migrations = ''

    platform = ''
    user_ip = ''
    user_agent = ''

    languages = {
        'en': 'English',
        'uk': 'Українська',
        'pl': 'Polski'
    }
    page_permission = ''
    proj_config = yaml.safe_load(open("../proj.yml"))['proj']

    db = ''          # НЕ використовуй як сесію
    db_driver = ''   # залишено для сумісності

    class_dict = {}
    connected_clients = {}
    system_settings = {}

    adm_menu_items = [
        {
            "title": "t_update_system",
            "icon": {"icon": "mdi mdi-cloud-download"},
            "children": [
                {"title": "t_install_new_components2", "to": "/components-add"},
                {"title": "t_installed_components2", "to": "/components-list"},
                {"title": "t_update_component", "to": "/components-update"},
                {"title": "t_lnSyncData", "to": "/components-update-metadata"},
                {"title": "t_lnSyncDataLicense", "to": "/components-license-key"},
                {"title": "t_lnComponentLicenseKeys", "to": "/additionals-components-license-key"}
            ]
        }
    ]
    adm_menu_items_category = [
        {
            "title": 't_update_system',
            "to": "/components-add",
            "icon": "k2",
            "sort": 100
        }
    ]

    # шляхи до файлів підключень
    path_db = f"../cfg/{proj_config}/db"
    path_db_user = f"../cfg/k2/db/users/"
    path_redis = f"../cfg/{proj_config}/redis"
    db_login = None
    static_files = []
    caching_enabled = None
    nginx_static = False

    file_lic_syncfusion_path = ''
    file_lic_aggrid_path = ''
    file_lic_stimulsoft_path = ''
    key_syncfusion = ''
    key_aggrid = ''
    key_stimulsoft = ''

    search_comp = [
        f"../cfg/{proj_config}/components",
        f"../usr/{proj_config}/components",
        f"components"
    ]

    app_root_path = ''
    template = 'app_stack'

    authorized_users = {}
    authorized_users_file = '../data/k2_cloud_erp/users/authorized_users.json'

    # ВАЖЛИВО: для сумісності з існуючим кодом (login_manager = K2.login_manager)
    login_manager = LoginManager()

    # -------------------- ініціалізація --------------------
    def __init__(self, *args, **kwargs):
        super().__init__()
        self.secur = K2Secur()
        self.path = K2Path()
        self.data = K2Datasync()
        self.settings = K2Settings()
        self.notifications = K2Notifications()

    # поточний користувач (без збереження у полі інстансу)
    @property
    def current_user_login(self):
        return current_user if self._is_authed() else None

    # -------------------- DB URI --------------------
    def init_db_uri(self):
        """Пошук і ініціалізація файлу з підключення до бази даних (основна БД)."""
        try:
            yml_conf = '/test_db.yml' if os.getenv("TESTING") else '/db.yml'
            db_file_path = self.path_db + yml_conf
            if os.path.exists(db_file_path) and os.path.getsize(db_file_path) > 0:
                with open(db_file_path, 'r') as f:
                    return yaml.load(f, Loader=yaml.FullLoader)
            return None
        except Exception as e:
            logging.error(f"init_db_uri: {e}")
            return None

    def init_db_uri_user(self):
        """Параметри підключення до БД для поточного користувача (db_login режим)."""
        try:
            uid = self.get_current_user().get_id()
            db_file_path = os.path.join(self.path_db_user, uid, 'db.yml')
            if os.path.exists(db_file_path) and os.path.getsize(db_file_path) > 0:
                with open(db_file_path, 'r') as f:
                    return yaml.load(f, Loader=yaml.FullLoader)
            return None
        except Exception as e:
            logging.error(f"init_db_uri_user: {e}")
            return None

    def init_db_uri_custom(self):
        """Зчитує всі підключення з файлу db_custom.yml у словник за ключем."""
        try:
            db_file_path = os.path.join(self.path_db, 'db_custom.yml')
            if os.path.exists(db_file_path) and os.path.getsize(db_file_path) > 0:
                with open(db_file_path, 'r') as f:
                    db_configs = yaml.load(f, Loader=yaml.FullLoader)
                if isinstance(db_configs, list):
                    return {cfg.get('key'): cfg for cfg in db_configs if cfg.get('key')}
            return None
        except Exception as e:
            logging.error(f"init_db_uri_custom: {e}")
            return None

    def init_db(self):
        """Повертає DB URI основної БД."""
        db_config = self.init_db_uri()
        if not db_config:
            return f"sqlite:///database.db"
        return (f"{db_config['driver']}://{db_config['user']}:{db_config['password']}"
                f"@{db_config['host']}:{db_config['port']}/{db_config['dbname']}")

    def init_db_user(self):
        """DB URI для користувача (режим db_login=True)."""
        from app import app
        if (hasattr(app, 'login_manager')
                and isinstance(app.login_manager, LoginManager)
                and current_user
                and K2.db_login):
            self.create_db_file_config_user()
            db_config = self.init_db_uri_user()
            if db_config:
                return (f"{db_config['driver']}://{db_config['user']}:{db_config['password']}"
                        f"@{db_config['host']}:{db_config['port']}/{db_config['dbname']}")
        return None

    def init_db_custom(self, key: str):
        """DB URI для кастомного підключення за ключем."""
        try:
            db_map = self.init_db_uri_custom()
            if db_map and key in db_map:
                cfg = db_map[key]
                return (f"{cfg['driver']}://{cfg['user']}:{cfg['password']}"
                        f"@{cfg['host']}:{cfg['port']}/{cfg['dbname']}")
            logging.error(f"init_db_custom: key '{key}' not found")
            return None
        except Exception as e:
            logging.error(f"init_db_custom error: {e}")
            return None

    # -------------------- керування ролями на рівні БД (PostgreSQL) --------------------
    def create_db_role(self, user_name, password):
        """Створення користувача на рівні БД (PostgreSQL)."""
        try:
            import psycopg2
            db_config = K2().init_db_uri()
            conn = psycopg2.connect(
                dbname=db_config['dbname'],
                user=db_config['user'],
                password=db_config['password'],
                host=db_config['host'],
                port=db_config['port']
            )
            cur = conn.cursor()
            cur.execute("SELECT 1 FROM pg_roles WHERE rolname = %s", (user_name,))
            if cur.fetchone():
                cur.close(); conn.close()
                return "Користувач вже існує"

            cur.execute(f"CREATE USER \"{user_name}\" WITH PASSWORD %s", (password,))
            cur.execute(f"GRANT ALL PRIVILEGES ON DATABASE \"{db_config['dbname']}\" TO \"{user_name}\"")
            cur.execute(f"ALTER USER \"{user_name}\" WITH SUPERUSER")
            conn.commit()
            cur.close(); conn.close()
            return "Користувач успішно створений"
        except Exception as e:
            logging.error(f"create_db_role: {e}")
            return "Помилка створення користувача"

    def drop_db_role(self, user_name):
        """Видалення користувача на рівні БД (PostgreSQL)."""
        try:
            import psycopg2
            db_config = K2().init_db_uri()
            conn = psycopg2.connect(
                dbname=db_config['dbname'],
                user=db_config['user'],
                password=db_config['password'],
                host=db_config['host'],
                port=db_config['port']
            )
            cur = conn.cursor()
            cur.execute("SELECT 1 FROM pg_roles WHERE rolname = %s", (user_name,))
            if not cur.fetchone():
                cur.close(); conn.close()
                return "Користувач не існує"

            cur.execute(f"REVOKE ALL PRIVILEGES ON ALL TABLES IN SCHEMA public FROM \"{user_name}\"")
            cur.execute(f"DROP OWNED BY \"{user_name}\"")
            cur.execute(f"DROP USER \"{user_name}\"")
            conn.commit()
            cur.close(); conn.close()
            return "Користувач успішно видалений"
        except Exception as e:
            logging.error(f"drop_db_role: {e}")
            return "Помилка видалення користувача"

    def edit_pass_db_role(self, user_name, new_password):
        """Зміна пароля користувача на рівні БД (PostgreSQL)."""
        try:
            import psycopg2
            db_config = K2().init_db_uri()
            conn = psycopg2.connect(
                dbname=db_config['dbname'],
                user=db_config['user'],
                password=db_config['password'],
                host=db_config['host'],
                port=db_config['port']
            )
            cur = conn.cursor()
            cur.execute("SELECT 1 FROM pg_roles WHERE rolname = %s", (user_name,))
            if not cur.fetchone():
                cur.close(); conn.close()
                return "Користувач не існує"
            cur.execute(f"ALTER USER \"{user_name}\" WITH PASSWORD %s", (new_password,))
            conn.commit()
            cur.close(); conn.close()
            return "Пароль успішно змінено"
        except Exception as e:
            logging.error(f"edit_pass_db_role: {e}")
            return "Помилка зміни пароля"

    # -------------------- db_login допоміжні --------------------
    def create_db_file_config_user(self):
        """Створює/оновлює YAML з параметрами підключення для користувача (db_login)."""
        try:
            uid = self.get_current_user().get_id()
            db_file_path = os.path.join(self.path_db_user, uid, 'db.yml')
            db_cfg = K2().init_db_uri()

            from app import app
            import hashlib
            if hasattr(app, 'login_manager') and isinstance(app.login_manager, LoginManager) and current_user.is_authenticated:
                from components.k2site.k2site.models import K2users
                db = self._get_db()
                u = db.session.query(K2users.login, K2users.password).filter_by(user_id=uid).first()
                if not u:
                    return
                login, password = u
                password = hashlib.md5(password.encode()).hexdigest()
            else:
                return

            db_dir = os.path.dirname(db_file_path)
            os.makedirs(db_dir, exist_ok=True)

            if not os.path.exists(db_file_path):
                with open(db_file_path, 'w') as f:
                    yaml.safe_dump({}, f)

            with open(db_file_path, 'r') as f:
                existing = yaml.safe_load(f) or {}

            existing.update({
                'dbname': db_cfg['dbname'],
                'driver': db_cfg['driver'],
                'user': login,
                'password': password,
                'host': db_cfg['host'],
                'port': db_cfg['port'],
            })
            with open(db_file_path, 'w') as f:
                yaml.safe_dump(existing, f)
        except Exception as e:
            logging.error(f"create_db_file_config_user: {e}")

    # def db_user_engine(self, db):
    #     """Перемикач engine під користувача (НЕ викликати на кожен запит!)."""
    #     uri = self.init_db_user()
    #     if not uri:
    #         return
    #     if current_user.get_login() not in db.engines:
    #         db.engine.dispose()
    #         db.engines[current_user.get_login()] = create_engine(uri)
    #     K2Obj.db = db.engines[current_user.get_login()]

    def db_custom_engine(self, db):
        """Підключення кастомних engine (разово)."""
        db_dict = self.init_db_uri_custom()
        if not db_dict:
            return
        for key in db_dict.keys():
            if key not in db.engines:
                db.engine.dispose()
                uri = self.init_db_custom(key)
                if uri:
                    db.engines[key] = create_engine(uri)

    # -------------------- LoginManager --------------------
    def init_lm(self, app):
        """Ініціалізація LoginManager (для сумісності з твоїм кодом)."""
        self.login_manager.init_app(app)

    # -------------------- request/auth helpers --------------------
    def _get_db(self):
        """Повертає екземпляр Flask‑SQLAlchemy, прив’язаний до поточного Flask app."""
        try:
            return current_app.extensions['sqlalchemy']
        except Exception as e:
            raise RuntimeError("No active Flask application context.") from e

    def _is_authed(self) -> bool:
        try:
            return has_request_context() and bool(getattr(current_user, "is_authenticated", False))
        except Exception:
            return False

    def get_current_user(self):
        """Повертає current_user або None (якщо немає request‑контексту)."""
        return current_user if self._is_authed() else None

    def _resolve_user_id(self, user_id: str | None) -> str | None:
        if user_id:
            return str(user_id)
        if self._is_authed():
            try:
                return str(current_user.get_id())
            except Exception:
                return None
        return None

    # -------------------- ЄДИНИЙ SQL на контекст користувача + request‑кеш --------------------
    def _get_user_context_db(self, user_id: str) -> dict | None:
        """
        Один запит, що повертає:
          - project_id, project_name
          - counterpart_id, counterpart_name
          - storage_id, storage_name
          - structural_division_id
          - user_theme (шаблон)
        """
        if not user_id:
            return None

        try:
            from components.k2site.k2site.models import K2users, K2Proj
            from components.k2handbook.k2handbook.models import k2counterparts, k2storage
        except Exception as e:
            logging.error(f"_get_user_context_db import models: {e}")
            return None

        db = self._get_db()
        P = aliased(K2Proj)
        C = aliased(k2counterparts)
        S = aliased(k2storage)

        row = (
            db.session.query(
                K2users.user_id,
                K2users.login,
                K2users.projid,
                K2users.counterpart_id,
                K2users.storage_id,
                K2users.structural_division_id,
                K2users.user_theme_pyt,        # <--- одразу підтягуємо тему
                P.projname,
                C.counterpart_name,
                S.storage_name,
            )
            .outerjoin(P, P.projid == K2users.projid)
            .outerjoin(C, C.counterpart_id == K2users.counterpart_id)
            .outerjoin(S, S.storage_id == K2users.storage_id)
            .filter(K2users.user_id == user_id)
            .first()
        )
        if not row:
            return None

        (_uid, login, projid, counterpart_id, storage_id, sdiv_id, user_theme,
         projname, counterpart_name, storage_name) = row

        return {
            "user_id": user_id,
            "login": login,
            "project_id": projid,
            "project_name": projname,
            "counterpart_id": counterpart_id,
            "counterpart_name": counterpart_name,
            "storage_id": storage_id,
            "storage_name": storage_name,
            "structural_division_id": sdiv_id,
            "user_theme": user_theme,
        }

    def _get_user_context(self, user_id: str | None = None) -> dict | None:
        uid = self._resolve_user_id(user_id)
        if not uid:
            return None
        # request‑кеш у g
        ctx = getattr(g, "_k2_ctx", None)
        if ctx and ctx.get("user_id") == uid:
            return ctx
        ctx = self._get_user_context_db(uid)
        if ctx:
            g._k2_ctx = ctx
        return ctx

    # -------------------- публічні геттери (без додаткових SQL) --------------------
    def get_user_project_id(self, user_id: str | None = None):
        ctx = self._get_user_context(user_id);  return ctx["project_id"] if ctx else None

    def get_user_project_name(self, user_id: str | None = None):
        ctx = self._get_user_context(user_id);  return (ctx["project_name"] or None) if ctx else None

    def get_user_counterparts_id(self, user_id: str | None = None):
        ctx = self._get_user_context(user_id);  return ctx["counterpart_id"] if ctx else None

    def get_user_counterparts_name(self, user_id: str | None = None):
        ctx = self._get_user_context(user_id);  return (ctx["counterpart_name"] or "-") if ctx else "-"

    def get_user_storage_id(self, user_id: str | None = None):
        ctx = self._get_user_context(user_id);  return ctx["storage_id"] if ctx else None

    def get_user_stoages_name(self, user_id: str | None = None):
        ctx = self._get_user_context(user_id);  return (ctx["storage_name"] or "-") if ctx else "-"

    def get_user_structural_division_id(self, user_id: str | None = None):
        ctx = self._get_user_context(user_id);  return ctx["structural_division_id"] if ctx else None

    def get_user_structural_division_id_tree(self, user_id: str | None = None):
        root_id = self.get_user_structural_division_id(user_id)
        if not root_id:
            return ()
        db = self._get_db()
        sql = """
        WITH RECURSIVE tree AS (
            SELECT sd.structural_division_id, sd.parentid
              FROM k2structural_divisions sd
             WHERE sd.structural_division_id = :root_id AND sd.active = 1
          UNION ALL
            SELECT ch.structural_division_id, ch.parentid
              FROM k2structural_divisions ch
              JOIN tree t ON ch.parentid = t.structural_division_id
             WHERE ch.active = 1
        )
        SELECT structural_division_id FROM tree;
        """
        rows = db.session.execute(text(sql), {"root_id": root_id}).fetchall()
        ids = tuple(r[0] for r in rows) if rows else ()
        return ids[0] if len(ids) == 1 else ids

    # -------------------- UI / меню / шаблон --------------------
    def get_project_setting(self):
        """Форма зміни проекту (виклик методу представлення)."""
        from app import app
        if hasattr(app, 'login_manager') and isinstance(app.login_manager, LoginManager) and current_user.is_authenticated:
            try:
                from components.k2site.k2site.views import K2Site
                return K2Site.show_projects()
            except Exception as e:
                logging.error(f"get_project_setting: {e}")
        return ''

    def search_class_dict(self, parent_class=None):
        """Рекурсивний пошук властивостей нащадків базового класу."""
        if parent_class is None:
            parent_class = K2Obj
        for subclass in parent_class.__subclasses__():
            class_name = subclass.__name__.lower()
            K2.class_dict[class_name] = {}
            if hasattr(subclass, '_path_class'):
                K2.class_dict[class_name]['path'] = subclass._path_class
            if hasattr(subclass, 'export_data'):
                K2.class_dict[class_name]['export_data'] = subclass.export_data
            if hasattr(subclass, '_name_file_class'):
                K2.class_dict[class_name]['name_file'] = subclass._name_file_class
            self.search_class_dict(subclass)

    def url_map(self):
        from flask import current_app as _app
        return [rule.rule for rule in _app.url_map.iter_rules()]

    def get_menu_url(self):
        """Список URL активних пунктів меню."""
        from .k2upd import K2Menu
        response = K2.menu
        K2.menu_url = K2Menu().get_admin_menu_url(response)

    def search_menu_items(self):
        """Завантаження меню адміністратора."""
        try:
            from .k2upd import K2Menu
            menu = K2Menu()
            K2.menu = menu.get_admin_menu()
        except Exception as e:
            logging.error(f"search_menu_items {e}")

    def search_menu_items_category(self):
        """Категорії меню адміністратора."""
        try:
            from k2.k2upd import K2Menu
            K2.menu_category = K2Menu().get_admin_menu_category()
        except Exception as e:
            logging.error(f"search_menu_items_category: {e}")

    def current_template(self):
        """Поточний шаблон користувача (без додаткових SQL)."""
        ctx = self._get_user_context()
        return ctx.get("user_theme") if ctx and ctx.get("user_theme") else self.template

    def search_static_files(self):
        components_names = K2.search_comp_names()
        _ = [
            subclass.adm_menu_items_category[0]
            for subclass in K2Obj.__subclasses__()
            if hasattr(subclass, 'adm_menu_items_category') and subclass.__name__.lower() in components_names
        ]

    @classmethod
    def namemenu(cls, url):
        """Генерує ідентифікатор сторінки за URL."""
        try:
            return 'k2itm' + url.replace('/', '-')
        except Exception:
            return 'k2itm'

    # -------------------- Ролі / права --------------------
    @classmethod
    def get_user_role(cls, user_id):
        """Отримання roleid користувача (з current_user якщо є, або один SQL)."""
        try:
            # швидкий шлях через current_user
            if has_request_context() and getattr(current_user, "is_authenticated", False):
                getter = getattr(current_user, "get_role_id", None)
                if callable(getter):
                    return getter()

            from components.k2site.k2site.models import K2users
            db = current_app.extensions['sqlalchemy']
            row = db.session.query(K2users.roleid).filter_by(user_id=user_id).first()
            return row[0] if row else None
        except Exception as ex:
            logging.error(f"get_user_role: {ex}")
            return None

    @classmethod
    def get_current_user_role_name(cls):
        """Отримання rolename поточного користувача (максимум 1 SQL)."""
        try:
            from components.k2site.k2site.models import K2roles
            user_id = current_user.get_id()
            roleid = cls.get_user_role(user_id)
            if not roleid:
                return {}
            db = current_app.extensions['sqlalchemy']
            row = db.session.query(K2roles.rolename).filter_by(roleid=roleid).first()
            return row[0] if row else {}
        except Exception as ex:
            logging.error(f"get_current_user_role_name: {ex}")
            return {}

    @classmethod
    def get_user_permissions(cls):
        """Права користувача для поточного URL (мінімум запитів)."""
        try:
            from components.k2site.k2site.models import K2users
            from .k2admmenu import K2admin_Menus_Prava, K2admin_menus

            user_id = current_user.get_id()
            if K2Secur().is_superadmin_cloud(current_user):
                return {'roleid': '-1000', 'r': 1, 'w': 1, 'i': 1, 'd': 1, 'c': 1, 'exp': 1, 'imp': 1,
                        'del_': 1, 'settable': 1, 'cutpast': 1, 'enable': 1, 'active': 1}

            user_role = cls.get_user_role(user_id)
            namemenu = K2.namemenu(request.path)

            db = current_app.extensions['sqlalchemy']
            perms = (
                db.session.query(K2admin_Menus_Prava)
                .join(K2admin_menus, K2admin_Menus_Prava.menuid == K2admin_menus.menuid)
                .filter(K2admin_Menus_Prava.roleid == user_role,
                        K2admin_menus.namemenu == namemenu,
                        K2admin_menus.active == 1)
                .options(joinedload(K2admin_Menus_Prava.menu))
                .all()
            )

            if not perms and request.args.get('parent'):
                parent = K2.namemenu('/' + request.args.get('parent')).replace('--', '-')
                perms = (
                    db.session.query(K2admin_Menus_Prava)
                    .join(K2admin_menus, K2admin_Menus_Prava.menuid == K2admin_menus.menuid)
                    .filter(K2admin_Menus_Prava.roleid == user_role,
                            K2admin_menus.namemenu == parent,
                            K2admin_menus.active == 1)
                    .options(joinedload(K2admin_Menus_Prava.menu))
                    .all()
                )

            if perms:
                p = perms[0]
                return {
                    'roleid': p.roleid, 'r': p.r, 'w': p.w, 'i': p.i, 'd': p.d, 'c': p.c,
                    'exp': p.exp, 'imp': p.imp, 'del_': p.del_, 'settable': p.settable,
                    'cutpast': p.cutpast, 'enable': p.enable, 'active': p.active
                }
            return {}
        except Exception as e:
            logging.error(f"get_user_permissions: {e}")
            return {}

    # -------------------- Пошук файлів конфіг/локалізації --------------------
    @classmethod
    def search_yml(cls, name_yml):
        """Шукає yml у 'yml' підпапках компонентів або у data‑папці."""
        search_data_folder = True
        for subclass in K2Obj.__subclasses__():
            caller_path = inspect.getfile(subclass)
            caller_dir = os.path.dirname(caller_path)
            yml_dir = os.path.join(caller_dir, 'yml')
            yml_file = os.path.join(yml_dir, f'{name_yml}.yml')
            if os.path.exists(yml_file):
                search_data_folder = False
                return caller_dir
        if search_data_folder:
            yml_path = f'yml/{name_yml}.yml'
            caller_dir = K2().path.find_data_file(yml_path)
            if os.path.exists(caller_dir):
                return caller_dir.replace(yml_path, '')
        return None

    @classmethod
    def get_path_to_root(cls, caller_file):
        curr_dir = os.path.dirname(__file__)
        root_dir = os.path.dirname(curr_dir)
        return os.path.relpath(caller_file, root_dir)

    @classmethod
    def get_path_abs(cls, caller_file):
        return os.path.abspath(caller_file)

    @classmethod
    def generate_id(cls):
        generated_id = uuid.uuid4().hex[:32]
        return ''.join([generated_id[i:i + 4] for i in range(0, len(generated_id), 4)])

    @classmethod
    def load_babel_translation_directories(cls):
        try:
            with open('languages/babel_translation_directories.yml', 'r') as f:
                data = yaml.safe_load(f)
            cls.babel_translation_directories = data.get('babel_translation_directories', 'languages')
        except Exception:
            cls.babel_translation_directories = 'languages'

    @classmethod
    def search_babel_translation_directories(cls):
        from k2.k2trans import K2Lang
        response = K2Lang.find_language()
        cls.babel_translation_directories = response or 'languages'
        with open('languages/babel_translation_directories.yml', 'w') as f:
            yaml.dump({'babel_translation_directories': cls.babel_translation_directories}, f)

    # -------------------- Мова --------------------
    @classmethod
    def get_locale(cls):
        if 'lang' in session:
            return session['lang']
        elif cls.current_language:
            return cls.current_language
        return cls.default_language

    @classmethod
    def get_locale_id(cls):
        try:
            from components.k2handbook.k2handbook.models import k2language
            db = current_app.extensions['sqlalchemy']
            lang = cls.get_locale()
            row = db.session.query(k2language.language_id).filter_by(language_symbol=lang).first()
            return row[0] if row else None
        except Exception as e:
            logging.error(f"get_locale_id: {e}")
            return None

    @classmethod
    def get_active_lang_list(cls):
        try:
            from components.k2handbook.k2handbook.models import k2language
            db = current_app.extensions['sqlalchemy']
            rows = db.session.query(k2language.language_symbol).filter_by(language_active=1).all()
            return [r[0] for r in rows] if rows else []
        except Exception as e:
            logging.error(f"get_active_lang_list: {e}")
            return []

    # -------------------- System settings --------------------
    @classmethod
    def create_system_settings(cls):
        data_settings = {
            "title": "K2 Cloud ERP",
            "description": "K2 Cloud ERP: WMS, CRM, Document Management, CMS, Education System, VDoc",
            "logo_path": "",
            "caching_enabled": False,
            "db_login": False,
            "default_language": "uk",
            "action_log": False
        }
        file_system_settings = '../cfg/k2/settings/system_settings.json'
        if not os.path.exists(file_system_settings):
            os.makedirs(os.path.dirname(file_system_settings), exist_ok=True)
            with open(file_system_settings, 'w') as file:
                json.dump(data_settings, file, indent=4, ensure_ascii=False)

    @classmethod
    def get_platform(cls):
        cls.platform = platform.system()
        return cls.platform

    # -------------------- components.yml --------------------
    @classmethod
    def ins_search_comp(self):
        result = {'components': []}
        for base in self.search_comp:
            path = os.path.join(base, "components.yml")
            if os.path.exists(path):
                with open(path, 'r') as file:
                    data = yaml.safe_load(file) or {}
                for comp in data.get('components', []):
                    name = comp.get('name')
                    if name and not any(c.get('name') == name for c in result['components']):
                        result['components'].append(comp)
        return result

    @classmethod
    def search_comp_names(self):
        data = self.ins_search_comp()
        comps = data.get('components', [])
        return [c['name'] for c in comps if c.get('installed', True)]

    # -------------------- діагностика потоків/грінлетів --------------------
    @classmethod
    def dump_stacks(cls):
        dump = []
        greenlet_count = 0
        import gc, threading, sys as _sys, traceback
        try:
            from greenlet import greenlet
        except ImportError:
            greenlet = None

        threads = {th.ident: th.name for th in threading.enumerate()}

        for thread, frame in _sys._current_frames().items():
            thread_name = threads.get(thread)
            if thread_name:
                dump.append('Thread: %s\n' % thread_name)
                dump.append(''.join(traceback.format_stack(frame)))
                dump.append('\n')

        if greenlet:
            for ob in gc.get_objects():
                try:
                    if isinstance(ob, greenlet) and ob:
                        greenlet_count += 1
                except Exception:
                    pass
        return greenlet_count

    # -------------------- утиліти --------------------
    @classmethod
    def compare_versions(cls, version1, version2):
        a = list(map(int, version1.split('.')))
        b = list(map(int, version2.split('.')))
        for i in range(min(len(a), len(b))):
            if a[i] < b[i]: return -1
            if a[i] > b[i]: return 1
        if len(a) < len(b): return -1
        if len(a) > len(b): return 1
        return 0

    # -------------------- логування повідомлень у файлі + Socket.IO --------------------
    log_error = 'ERROR'
    log_warning = 'WARNING'
    log_success = 'SUCCESS'

    @classmethod
    def log_entry(cls, error_id, status, message, name):
        return {
            'error_id': error_id,
            'date': datetime.now().strftime("[%Y-%m-%d %H:%M:%S]"),
            'status': status,
            'page': name,
            'message': message,
            'short_message': message[:100] + "..." if len(message) > 100 else message + "..."
        }

    @classmethod
    def _load_authorized(cls):
        """Безпечне читання списку авторизованих користувачів із JSON."""
        directory = os.path.dirname(cls.authorized_users_file)
        os.makedirs(directory, exist_ok=True)
        if not os.path.exists(cls.authorized_users_file):
            with open(cls.authorized_users_file, 'w') as f:
                json.dump([], f)
            return []
        try:
            with open(cls.authorized_users_file, 'r') as f:
                return json.load(f)
        except JSONDecodeError as ex:
            logging.error(f"Corrupted JSON in {cls.authorized_users_file}: {ex}")
            backup = cls.authorized_users_file + '.broken'
            shutil.move(cls.authorized_users_file, backup)
            with open(cls.authorized_users_file, 'w') as f:
                json.dump([], f)
            return []

    @classmethod
    def _save_authorized(cls, data):
        tmp = cls.authorized_users_file + '.tmp'
        with open(tmp, 'w') as f:
            json.dump(data, f, indent=4)
        os.replace(tmp, cls.authorized_users_file)

    def add_authorized_users(self, user_id, login):
        data = self._load_authorized()
        if any(u['user_id'] == user_id for u in data):
            return
        data.append({'user_id': user_id, 'login': login})
        try:
            self._save_authorized(data)
        except Exception as ex:
            logging.error(f"Не вдалося зберегти {self.authorized_users_file}: {ex}")

    def dell_authorized_users(self, user_id):
        data = self._load_authorized()
        new_data = [u for u in data if u['user_id'] != user_id]
        try:
            self._save_authorized(new_data)
        except Exception as ex:
            logging.error(f"Не вдалося зберегти {self.authorized_users_file}: {ex}")

    @classmethod
    def clear_logging_messages(cls):
        try:
            logging_dir = os.path.dirname(K2().path.data_path('conf/error_log/', True))
            file_path = os.path.join(logging_dir, "error.json")
            with open(file_path, 'w') as file:
                json.dump([], file, indent=4)
        except Exception:
            pass

    # @classmethod
    # def save_logging_message(cls, error_id, status, message, name):
    #     logging_dir = os.path.dirname(K2().path.data_path('conf/error_log/', True))
    #     os.makedirs(logging_dir, exist_ok=True)
    #     filename = "error.json"
    #     logging_file = os.path.join(logging_dir, filename)
    #
    #     log_entry = cls.log_entry(error_id, status, message, name)
    #
    #     if os.path.exists(logging_file):
    #         with open(logging_file, 'r') as f:
    #             logs = json.load(f)
    #     else:
    #         logs = []
    #     logs.append(log_entry)
    #     with open(logging_file, 'w') as f:
    #         json.dump(logs, f, indent=4)
    #
    #     from app import socketio
    #     error_dict = cls.load_logging_messages(name)
    #     pageid = K2.namemenu(name)
    #     uid = current_user.get_id()
    #     if uid in K2.connected_clients:
    #         sid = K2.connected_clients[uid]['sid']
    #         socketio.emit(f'update_errors_count_result_{pageid}',
    #                       {'error_count': error_dict['error_count'], 'warning_count': error_dict['warning_count']},
    #                       room=sid)
    #     else:
    #         socketio.emit(f'update_errors_count_result_{pageid}',
    #                       {'error_count': error_dict['error_count'], 'warning_count': error_dict['warning_count']},
    #                       room=uid)

    @classmethod
    def save_logging_message(cls, error_id, status, message, name):
        """
        Зберігає лог у JSON-файл і надсилає оновлення через SocketIO усім активним сесіям користувача.
        """
        logging_dir = os.path.dirname(K2().path.data_path('conf/error_log/', True))
        os.makedirs(logging_dir, exist_ok=True)
        filename = "error.json"
        logging_file = os.path.join(logging_dir, filename)

        log_entry = cls.log_entry(error_id, status, message, name)

        if os.path.exists(logging_file):
            with open(logging_file, 'r') as f:
                logs = json.load(f)
        else:
            logs = []
        logs.append(log_entry)
        with open(logging_file, 'w') as f:
            json.dump(logs, f, indent=4)

        # Завантажити кількість помилок
        error_dict = cls.load_logging_messages(name)
        pageid = K2.namemenu(name)
        uid = current_user.get_id()
        from app import socketio
        from components.k2site.k2site.objects.presence.presence_service import active_sessions

        # 🔁 Отримуємо усі активні сесії користувача через Redis presence
        sessions = active_sessions(uid)
        payload = {
            'error_count': error_dict['error_count'],
            'warning_count': error_dict['warning_count']
        }

        if sessions:
            for sess in sessions:
                sid = sess.get("session_id") or sess.get("sid")
                if sid:
                    socketio.emit(
                        f'update_errors_count_result_{pageid}',
                        payload,
                        room=sid
                    )
        else:
            socketio.emit(
                f'update_errors_count_result_{pageid}',
                payload,
                room=uid
            )

    @classmethod
    def load_logging_messages(cls, page_url: str):
        logging_dir = os.path.dirname(K2().path.data_path('conf/error_log/', True))
        logging_file = os.path.join(logging_dir, "error.json")
        if not os.path.exists(logging_file):
            return {'error_messages': '', 'error_count': 0, 'warning_count': 0}

        with open(logging_file, 'r') as f:
            logs = json.load(f)

        filtered = [log for log in logs if log['page'] == str(page_url)]
        formatted = []
        error_count = 0
        warning_count = 0

        for log in reversed(filtered):
            status = log.get('status')
            if status == cls.log_error:
                error_count += 1
                log_class = 'e-error'
            elif status == cls.log_warning:
                warning_count += 1
                log_class = 'e-warning'
            elif status == cls.log_success:
                warning_count += 1  # якщо так і задумано — лишаю
                log_class = 'e-success'
            else:
                log_class = ''

            formatted.append(
                f"""
    <div class="log-entry" data-error-id="{log['error_id']}">
        <span class="mdi mdi-alert e-{status.lower()}"></span>
        <span>{log['date']}</span> -
        <span class="{log_class}">{status.upper()}</span> -
        <span class="log-entry-text">"{log['short_message']}"</span>
        <button class="log-more-btn"
                data-error-id="{log['error_id']}"
                style="border:none; background:none; cursor:pointer; padding:0 4px; display:inline-flex; align-items:center;">
            <i class="mdi mdi-dots-horizontal" style="font-size:16px; color:#0d6efd;"></i>
        </button>
    </div>
                """.strip()
            )

        return {
            'error_messages': formatted,
            'error_count': error_count,
            'warning_count': warning_count
        }

    @classmethod
    def load_logging_message_by_id(cls, error_id: str):
        logging_dir = os.path.dirname(K2().path.data_path('conf/error_log/', True))
        logging_file = os.path.join(logging_dir, "error.json")
        if not os.path.exists(logging_file):
            return None
        with open(logging_file, 'r') as f:
            logs = json.load(f)
        for log in logs:
            if log['error_id'] == error_id:
                return log
        return None

    # @classmethod
    # def logging_message(cls, status, message, page_url=None, show_message=True):
    #     from app import socketio
    #     if page_url:
    #         name = cls.namemenu(page_url)
    #     else:
    #         client_data = K2.connected_clients.get(current_user.get_id())
    #         print(client_data)
    #         if client_data:
    #             page_url = client_data['page_url'].replace('//', '/')
    #             name = cls.namemenu(page_url)
    #         else:
    #             name = None
    #
    #     status = status.upper()
    #     error_id = K2.generate_id()
    #     cls.save_logging_message(error_id, status, message, page_url)
    #
    #     if status == cls.log_error:
    #         logging.error(message)
    #     elif status == cls.log_warning:
    #         logging.warning(message)
    #     elif status == cls.log_success:
    #         logging.info(message)
    #
    #     log_entry = cls.log_entry(error_id, status, message, page_url)
    #     uid = current_user.get_id()
    #     if uid in K2.connected_clients:
    #         sid = K2.connected_clients[uid]['sid']
    #         if name:
    #             socketio.emit(f'update_errors_message_result_{name}', log_entry, room=sid)
    #         else:
    #             socketio.emit(f'update_errors_message_result', log_entry, room=sid)
    #     else:
    #         if name:
    #             socketio.emit(f'update_errors_message_result_{name}', log_entry, room=uid)
    #         else:
    #             socketio.emit(f'update_errors_message_result', log_entry, room=uid)
    #
    #     if show_message:
    #         if uid in K2.connected_clients:
    #             sid = K2.connected_clients[uid]['sid']
    #             if name:
    #                 socketio.emit(f'show_logging_message_{name}', {'status': status, 'message': message}, room=sid)
    #             else:
    #                 socketio.emit(f'show_logging_message_', {'status': status, 'message': message}, room=sid)
    #         else:
    #             if name:
    #                 socketio.emit(f'show_logging_message_{name}', {'status': status, 'message': message}, room=uid)
    #             else:
    #                 socketio.emit(f'show_logging_message_', {'status': status, 'message': message}, room=uid)
    #
    #     # -------------------- імпорт/міграції CSV у таблиці довідників --------------------

    @classmethod
    def logging_message(cls, status, message, page_url=None, show_message=True):
        """
        Створює лог і розсилає повідомлення через SocketIO усім активним сесіям користувача.
        """
        uid = current_user.get_id()
        if page_url:
            name = cls.namemenu(page_url)
        else:
            name = None

        status = status.upper()
        error_id = K2.generate_id()
        cls.save_logging_message(error_id, status, message, page_url)

        # Визначення рівня логування
        if status == cls.log_error:
            logging.error(message)
        elif status == cls.log_warning:
            logging.warning(message)
        elif status == cls.log_success:
            logging.info(message)

        log_entry = cls.log_entry(error_id, status, message, page_url)
        from app import socketio
        from components.k2site.k2site.objects.presence.presence_service import active_sessions
        # 🔁 Отримати всі сесії користувача
        sessions = active_sessions(uid)
        event_suffix = f'_{name}' if name else ''
        event_message = f'update_errors_message_result{event_suffix}'
        event_show = f'show_logging_message{event_suffix}'

        if sessions:
            for sess in sessions:
                sid = sess.get("session_id") or sess.get("sid")
                if not sid:
                    continue
                socketio.emit(event_message, log_entry, room=sid)
                if show_message:
                    socketio.emit(event_show, {'status': status, 'message': message}, room=sid)
        else:
            socketio.emit(event_message, log_entry, room=uid)
            if show_message:
                socketio.emit(event_show, {'status': status, 'message': message}, room=uid)

    def apply_csv_files_migrations(self, component_name: str):
        """ Імпорт CSV у таблиці довідників для компонента. """
        results = {}
        db = self._get_db()

        try:
            migration_path = Path(f"components/{component_name}/{component_name}/data/file_migrations").resolve()
            if not migration_path.exists():
                return jsonify({'result': 'error', 'message': f'Migration path not found: {migration_path}'})

            for filename in os.listdir(migration_path):
                if not filename.endswith('.csv'):
                    continue

                table_name = filename[:-4]
                filepath = migration_path / filename

                with open(filepath, newline='', encoding='utf-8') as csvfile:
                    reader = csv.DictReader(csvfile)
                    rows = list(reader)
                    if not rows:
                        continue

                    columns = reader.fieldnames or []

                    if not columns:
                        logging.warning(f"No columns found in {filename}")
                        continue

                    id_column = columns[0]
                    logging.info(f"Using first column '{id_column}' as ID for table '{table_name}'")

                    inserted = 0
                    skipped = 0
                    now = datetime.now()

                    # Визначаємо, які колонки потрібно додати (якщо їх ще немає)
                    additional_columns = []
                    if 'active' not in columns:
                        additional_columns.append('active')
                    if 'createdate' not in columns:
                        additional_columns.append('createdate')
                    if 'updatedate' not in columns:
                        additional_columns.append('updatedate')

                    insert_columns = columns + additional_columns
                    placeholders = ', '.join([f':{col}' for col in insert_columns])
                    col_names = ', '.join(insert_columns)

                    for row in rows:
                        row_id = (row.get(id_column) or "").strip()
                        if not row_id:
                            logging.warning(f"Empty ID value in row, skipping: {row}")
                            skipped += 1
                            continue

                        # Перевіряємо чи існує запис з таким ID
                        exists = db.session.execute(
                            text(f"SELECT 1 FROM {table_name} WHERE {id_column} = :id"),
                            {"id": row_id}
                        ).fetchone()

                        if exists:
                            logging.info(f"Record with {id_column}={row_id} already exists, skipping")
                            skipped += 1
                            continue

                        insert_values = {}
                        for col in columns:
                            value = row.get(col) or ""
                            insert_values[col] = value.strip()

                        # Додаємо тільки ті додаткові колонки, яких немає в CSV
                        if 'active' not in columns:
                            insert_values['active'] = 1
                        if 'createdate' not in columns:
                            insert_values['createdate'] = now
                        if 'updatedate' not in columns:
                            insert_values['updatedate'] = now

                        db.session.execute(
                            text(f"INSERT INTO {table_name} ({col_names}) VALUES ({placeholders})"),
                            insert_values
                        )
                        inserted += 1
                        logging.info(f"Inserted record with {id_column}={row_id}")

                    results[table_name] = {
                        'inserted': inserted,
                        'skipped': skipped,
                        'id_column_used': id_column,
                        'additional_columns_added': additional_columns
                    }
                    logging.info(
                        f"Table {table_name}: inserted {inserted}, skipped {skipped}, added columns: {additional_columns}")

            db.session.commit()
            return jsonify({'result': 'success', 'details': results})

        except Exception as ex:
            db.session.rollback()
            logging.error(f"{ex} - error during handbook migration")
            return jsonify({'result': 'error', 'message': str(ex)})



    # -------------------- інжекція додаткових властивостей із конфігів --------------------
    path_objs = [
        f"../cfg/{proj_config}/k2/{name.lower()}.py",
        f"cfg/{name.lower()}.py",
        f"usr/cfg/{name.lower()}.py"
    ]



    # ----------- системні файли/ліцензії (акуратно) -----------
    file_lic_path = '../cfg/k2/license/key.yml'
    try:
        with open(file_lic_path, 'r') as file:
            data = yaml.safe_load(file) or {}
        update_token = data.get('license_key', '')
    except Exception:
        update_token = ''

    file_system_settings = '../cfg/k2/settings/system_settings.json'
    try:
        with open(file_system_settings, 'r') as file:
            system_settings_dict = json.load(file) or {}
    except Exception:
        system_settings_dict = {
            "title": "K2 Cloud ERP",
            "description": "",
            "logo_path": "",
            "caching_enabled": False,
            "db_login": False,
            "default_language": "uk",
            "action_log": False
        }
    system_settings = system_settings_dict
    default_language = system_settings.get('default_language', 'uk')
    db_login = system_settings.get('db_login', False)
    caching_enabled = system_settings.get('caching_enabled', False)

    file_lic_syncfusion_path = '../cfg/k2/license/syncfusion.yml'
    try:
        with open(file_lic_syncfusion_path, 'r') as file:
            data = yaml.safe_load(file) or {}
        key_syncfusion = data.get('license_key', '')
    except Exception:
        key_syncfusion = ''

    file_lic_aggrid_path = '../cfg/k2/license/aggrid.yml'
    try:
        with open(file_lic_aggrid_path, 'r') as file:
            data = yaml.safe_load(file) or {}
        key_aggrid = data.get('license_key', '')
    except Exception:
        key_aggrid = ''

    file_lic_stimulsoft_path = '../cfg/k2/license/stimulsoft.yml'
    try:
        with open(file_lic_stimulsoft_path, 'r') as file:
            data = yaml.safe_load(file) or {}
        key_stimulsoft = data or {}
    except Exception:
        key_stimulsoft = {}

    try:
        port_config = yaml.safe_load(open("../proj.yml")).get('port')
        port = port_config if port_config is not None else '7654'
    except Exception:
        port = '7654'

    # Базовий домен локально
    domain_loc = f"http://127.0.0.1:{port}"

    # встановлені компоненти
    @staticmethod
    def component_list():
        try:
            with open("components/components.yml", 'r') as file:
                components_data = yaml.safe_load(file) or {}
            comps = components_data.get('components', [])
            return [{c['id']: c['name']} for c in comps if c.get('installed', False)]
        except Exception:
            return []

    components_install = component_list()

    # -------------------- онлайн‑користувачі / сесії --------------------
    # def get_authorized_users(self):
    #     if os.path.exists(self.authorized_users_file):
    #         with open(self.authorized_users_file, 'r') as f:
    #             return json.load(f)
    #     return []

    def check_logout_users(self, user_id):
        try:
            from app import app
            if hasattr(app, 'login_manager') and isinstance(app.login_manager, LoginManager) and current_user.is_authenticated:
                from components.k2site.k2site.models import K2users
                uid = current_user.get_id()

                # список активних
                user_online_list = [u['user_id'] for u in self.get_authorized_users()]

                db = self._get_db()
                u = db.session.query(K2users.locale, K2users.is_logout_by_admin).filter_by(user_id=uid).first()
                if u:
                    loc, is_out = u
                    K2.current_language = loc or K2.default_language
                    if uid not in user_online_list and is_out == 1:
                        logout_user()
                    else:
                        self.add_authorized_users(uid, getattr(current_user, "get_login", lambda: "")())
        except Exception as e:
            logging.info(e)

    def check_authorized_users(self, user_id):
        """Перевірити активність користувача (останній запис у журналі авторизацій)."""
        from app import app
        if hasattr(app, 'login_manager') and isinstance(app.login_manager, LoginManager) and current_user.is_authenticated:
            try:
                from components.k2site.k2site.models import K2log_auth
                db = self._get_db()
                last = db.session.query(K2log_auth.date_auth).filter_by(user_id=user_id).order_by(desc(K2log_auth.date_auth)).first()
                if last and last[0]:
                    _ = datetime.now() - last[0]
            except Exception as e:
                logging.error(f'check_authorized_users {e}')

    def init_redis_uri(self):
        """Пошук і ініціалізація файлу з параметрами підключення до Redis."""
        try:
            yml_conf = '/test_redis.yml' if os.getenv("TESTING") else '/redis.yml'
            redis_file_path = self.path_redis + yml_conf
            if os.path.exists(redis_file_path) and os.path.getsize(redis_file_path) > 0:
                with open(redis_file_path, 'r') as f:
                    return yaml.load(f, Loader=yaml.FullLoader)
            return None
        except Exception as e:
            logging.error(f"init_redis_uri: {e}")
            return None

    def redis_uri_string(self):
        """
        Формує рядок URI для підключення до Redis, використовуючи init_redis_uri().
        Підходить для Flask-SocketIO message_queue.
        """
        try:
            conf = self.init_redis_uri()

            # Якщо вже рядок — просто повертаємо
            if isinstance(conf, str):
                return conf

            # Якщо словник — збираємо URI
            if isinstance(conf, dict):
                host = conf.get("host", "localhost")
                port = conf.get("port", 6379)
                db = conf.get("db", 0)
                password = conf.get("password", "k2adm123456")

                if password:
                    return f"redis://:{password}@{host}:{port}/{db}"
                else:
                    return f"redis://{host}:{port}/{db}"

            # Якщо нічого не знайдено
            return "redis://localhost:6379/0"

        except Exception as e:
            logging.error(f"redis_uri_string: {e}")
            return "redis://localhost:6379/0"

      # той самий, який ми ініціалізували в create_app()

    # def get_online_user_ids(self) -> List[str]:
    #     """
    #     Повертає список user_id (str) хто онлайн зараз.
    #     Працює і з RedisPresenceStore, і з MemoryPresenceStore.
    #     """
    #     from components.k2site.k2site.objects.presence_store.presence_store import PresenceStore
    #     ids = PresenceStore.online_user_ids()  # set[str]
    #     # ВАЖЛИВО: ваші user_id у БД можуть бути int; тут повертаємо str,
    #     # а вже у виклику .in_() ви можете кастити до потрібного типу.
    #     return list(ids)
    #
    # def get_authorized_users_payload(self) -> List[Dict]:
    #     """
    #     Повертає список словників {"user_id": <id>} — рівно те, що очікує ваш існуючий код.
    #     """
    #     return [{"user_id": uid} for uid in self.get_online_user_ids()]

    # def get_authorized_users(self):
    #     """
    #     Сумісний інтерфейс: повертає [{"user_id": ...}, ...] для онлайн-користувачів.
    #     Джерелом виступає наша presence-система (Redis або in-memory).
    #     """
    #     try:
    #         return self.get_authorized_users_payload()
    #     except Exception as e:
    #         current_app.logger.warning("get_authorized_users fallback: %s", e)
    #         return []

    def get_authorized_users(self):
        try:
            from components.k2site.k2site.objects.presence.presence_service import online_user_ids
            # рівно те, що очікує get_main_dashboards_user_online()
            return [{"user_id": uid} for uid in online_user_ids()]
        except:
            return []

    @classmethod
    def get_connected_clients(cls, user_id):
        from k2.components.k2site.k2site.objects.presence.presence_service import (
            get_last_active_client_redis,
        )

        uid = str(user_id)
        data = get_last_active_client_redis(uid)

        if data and isinstance(data, dict):
            return data

        now = int(time.time())
        return {
            "sid": uid,
            "page_url": None,
            "ts": now,
            "_fallback_sid": True,
        }

    @classmethod
    def search_class_prop(cls, path_objs):
        code_str = None
        for path_obj in path_objs:
            if os.path.exists(path_obj):
                with open(path_obj, 'r') as file:
                    code_str = file.read()
                break
        else:
            sys.exit()
        return code_str

    # виконуємо інжекцію одразу (як було в оригіналі)
    code_str = K2Obj.search_class_prop(path_objs)
    try:
        exec(code_str)
    except Exception as e:
        print(f"Error in K2cfg {e}")
        sys.exit()

