import hashlib
import uuid
import os
import time
from sqlalchemy import text
import yaml
from pathlib import Path
import pytest
from pytest_docker_tools import container
import subprocess
import sys
import platform
from sqlalchemy import create_engine
from sqlalchemy.orm import sessionmaker, scoped_session
from unittest.mock import MagicMock
from pytz import timezone


def check_docker():
    """Перевіряє, чи Docker працює. Пропонує запустити Docker або встановити його, якщо він не знайдений."""
    # Якщо NO_MIGRATE=1, Docker не потрібен
    if os.getenv("NO_MIGRATE") == "1":
        print("⚠️  NO_MIGRATE=1: Пропускаємо перевірку Docker")
        return True

    try:
        # Виконуємо команду `docker info` для перевірки стану Docker
        result = subprocess.run(
            ["docker", "info"], stdout=subprocess.PIPE, stderr=subprocess.PIPE
        )
        if result.returncode == 0:
            print("✅ Docker працює!")
            return True
        else:
            print("❌ Docker не працює. Запуcтіть застосунок Docker.", result.stderr.decode())
            return False
    except FileNotFoundError:
        print("❌ Команда `docker` не знайдена.")
        print("Бажаєте перейти до сторінки завантаження Docker? (y/n)")
        choice = input().strip().lower()
        if choice == "y":
            if platform.system() == "Windows":
                print("🚀 Відкриваю сторінку завантаження Docker Desktop для Windows...")
                subprocess.run(["start", "https://www.docker.com/products/docker-desktop"], shell=True)
            elif platform.system() == "Linux":
                print("🚀 Відкриваю сторінку встановлення Docker Engine для Linux...")
                subprocess.run(["xdg-open", "https://docs.docker.com/engine/install/"])
            else:
                print("❌ Непідтримувана операційна система.")
            print("❌ Тести не можуть продовжитися без Docker.")
        else:
            print("❌ Тести не можуть продовжитися без Docker.")
        return False


# Перевірка наявності Docker перед створенням контейнера
if not check_docker():
    sys.exit(1)

is_no_migrate = os.getenv("NO_MIGRATE") == "1"
# Створення контейнера PostgreSQL тільки якщо NO_MIGRATE != 1
if is_no_migrate != "1":
    postgres_container = container(
        image="postgres:latest",
        scope="session",
        ports={"5432/tcp": 10101},
        environment={
            "POSTGRES_USER": "sadmin",
            "POSTGRES_PASSWORD": "k2adm123456",
            "POSTGRES_DB": "K2test_db",
        },
    )
else:
    # Мок для postgres_container коли NO_MIGRATE=1
    postgres_container = None

os.environ["TESTING"] = "1"


def write_db_config_to_yaml(container_info, file_path):
    """Записує конфігурацію бази даних у YAML-файл."""
    os.makedirs(os.path.dirname(file_path), exist_ok=True)
    db_config = {
        "dbname": container_info["dbname"],
        "driver": "postgresql",
        "host": container_info["host"],
        "password": container_info["password"],
        "port": str(container_info["port"]),
        "user": container_info["user"],
    }
    with open(file_path, "w") as yaml_file:
        yaml.dump(db_config, yaml_file, default_flow_style=False)
    time.sleep(1)


@pytest.fixture(scope="session", autouse=True)
def db_config(request):
    """Фікстура для отримання параметрів підключення до PostgreSQL і запису файлу конфігурації."""
    if os.getenv("NO_MIGRATE") == "1":
        print("⚠️  NO_MIGRATE=1: Пропускаємо створення БД конфігурації")
        return None

    postgres_cont = request.getfixturevalue("postgres_container")

    # Перевірка, що контейнер доступний. Якщо ні - тест впаде з помилкою.
    if not postgres_cont:
        pytest.fail(
            "❌ Не вдалося отримати фікстуру postgres_container. "
            "Перевірте, чи Docker запущено і працює коректно."
        )

    port = postgres_cont.ports["5432/tcp"][0]
    config = {
        "dbname": "K2test_db",
        "host": "localhost",
        "password": "k2adm123456",
        "port": port,
        "user": "sadmin",
        "driver": "postgresql",
    }

    current_file_path = Path(__file__)
    config_file_path = current_file_path.parent.parent / "cfg" / "k2" / "db" / "test_db.yml"
    write_db_config_to_yaml(config, config_file_path)
    print(f"📝 Конфігурацію бази даних записано у {config_file_path}")

    return config


@pytest.fixture(scope="session", autouse=True)
def setup_database(db_config):
    """Фікстура для виклику міграцій із окремою тестовою директорією міграцій."""
    if is_no_migrate:
        print("⚠️  NO_MIGRATE=1: Пропускаємо міграції бази даних")
        yield None
        return

    if not db_config:
        print("❗️ Пропуск тесту: фікстура db_config порожня.")
        yield None
        return

    from datetime import datetime
    import shutil
    import subprocess
    from pathlib import Path

    print("🚀 Підготовка тестової бази даних...")

    os.environ["FLASK_APP"] = "k2.app"
    os.environ["DATABASE_URL"] = (f"{db_config['driver']}://{db_config['user']}:{db_config['password']}@"
                                  f"{db_config['host']}:{db_config['port']}/{db_config['dbname']}")

    # Створюємо окрему директорію для тестових міграцій
    test_migrations_dir = Path("migrations_test")
    if test_migrations_dir.exists():
        shutil.rmtree(test_migrations_dir)
    test_migrations_dir.mkdir()

    # Копіюємо структуру і скрипти з основної директорії міграцій
    main_migrations_dir = Path("migrations")
    if main_migrations_dir.exists():
        # Копіюємо файли налаштувань alembic, але не копіюємо versions
        for item in main_migrations_dir.glob("*"):
            if item.name != "versions":
                if item.is_dir():
                    shutil.copytree(item, test_migrations_dir / item.name)
                else:
                    shutil.copy2(item, test_migrations_dir / item.name)

        # Створюємо нову директорію versions
        test_versions_dir = test_migrations_dir / "versions"
        test_versions_dir.mkdir(exist_ok=True)
    else:
        print("⚠️ Основна директорія міграцій не знайдена. Створюю нову.")
        subprocess.run(
            ["flask", "db", "init", "--directory", str(test_migrations_dir)],
            check=False, capture_output=True
        )

    # Встановлюємо MIGRATION_DIRECTORY для flask db
    os.environ["MIGRATION_DIRECTORY"] = str(test_migrations_dir)

    # Створюємо і застосовуємо початкову міграцію
    print("📦 Створюю початкову міграцію для тестової бази...")
    migration_result = subprocess.run(
        ["flask", "db", "migrate", "--directory", str(test_migrations_dir), "-m", "initial_test_migration"],
        check=False, capture_output=True
    )

    if migration_result.returncode == 0:
        print("✅ Міграція створена успішно")
        upgrade_result = subprocess.run(
            ["flask", "db", "upgrade", "--directory", str(test_migrations_dir)],
            check=False, capture_output=True
        )
        if upgrade_result.returncode == 0:
            print("✅ Міграції застосовані успішно")
        else:
            print(f"❌ Помилка при застосуванні міграцій: {upgrade_result.stderr.decode()}")
    else:
        print(f"❌ Помилка при створенні міграції: {migration_result.stderr.decode()}")

    yield test_migrations_dir

    # Прибираємо після тестів
    if test_migrations_dir.exists():
        try:
            shutil.rmtree(test_migrations_dir)
            print("🧹 Видалено тестову директорію міграцій")
        except Exception as e:
            print(f"❌ Помилка при видаленні тестової директорії міграцій: {e}")


@pytest.fixture(scope="function")
def db_connection(db_config):
    """Фікстура для підключення до бази даних."""
    if is_no_migrate:
        print("⚠️  NO_MIGRATE=1: Повертаємо мок замість справжнього підключення до БД")
        mock_session = MagicMock()
        yield mock_session
        return

    if not db_config:
        yield None
        return

    database_url = (f"{db_config['driver']}://{db_config['user']}:"
                    f"{db_config['password']}@{db_config['host']}:{db_config['port']}/{db_config['dbname']}")
    engine = create_engine(database_url, pool_pre_ping=True)

    session = scoped_session(sessionmaker(bind=engine))

    print("\n✅ Підключено до бази даних через SQLAlchemy!")
    try:
        yield session
    finally:
        session.remove()
        engine.dispose()
        print("\n✅ З'єднання з базою даних закрито.")


@pytest.fixture(scope="session", autouse=True)
def client_create(setup_database):
    """Фікстура для створення тестового клієнта Flask та запуску data-migrations."""
    from k2.app import app
    with app.test_client() as client:
        if not is_no_migrate:
            print('🚀 Виконую міграції даних для компонент...')
            response = client.get("/k2site/data-migrations", follow_redirects=True)
            assert response.status_code == 200, "❌ Помилка при виконанні data-migrations"
        else:
            print("⚠️  NO_MIGRATE=1: Пропускаємо data-migrations")
        yield client


@pytest.fixture
def ensure_sadmin_exists(db_connection):
    """Фікстура для створення користувача sadmin."""
    if is_no_migrate:
        print("⚠️  NO_MIGRATE=1: Пропускаємо створення користувача sadmin")
        return

    if not db_connection:
        return

    result = db_connection.execute(text("SELECT 1 FROM k2users WHERE login = 'sadmin' LIMIT 1"))
    if result.first() is None:
        user_id = hashlib.md5(str(uuid.uuid4()).encode()).hexdigest()
        password = hashlib.md5("k2adm123456".encode()).hexdigest()
        # ВСТАВ дані — заміни users_group_id і projid на конкретні або параметризуй
        projid = 'k2_cloud_erp'
        users_group_id = db_connection.execute(
            text("select users_group_id from k2users_group where name_users_group = 'k2'")).fetchone()[0]
        insert_sql = text(f"""
                INSERT INTO k2users (user_id, login, users_group_id, projid, roleid, name, email, password, active)
                VALUES ('{user_id}', 'sadmin', '{users_group_id}', '{projid}', '-1000', 'sadmin', '', '{password}', 1)
            """)
        db_connection.execute(insert_sql)
        db_connection.commit()


@pytest.fixture
def client_login(client_create, ensure_sadmin_exists):
    """Фікстура для логіну тестового клієнта."""
    if is_no_migrate:
        print("⚠️  NO_MIGRATE=1: Повертаємо клієнт без логіну")
        yield client_create
        return

    print('🔑 Логіню тестового користувача...')
    login = "sadmin"
    password = "k2adm123456"
    response = client_create.post(
        "/k2site/",
        data={"login": login, "psw": password, "roles": "-1000"},
        follow_redirects=True,
    )
    assert response.status_code == 200

    # Перевірка, що сесія активна, якщо потрібно
    home_response = client_create.get("/k2site/login-app", follow_redirects=True)

    yield client_create


@pytest.fixture
def storage_manager():
    """
    Фікстура для ініціалізації StorageManager
    """
    from k2.components.k2tests.k2tests.components_test_func.k2base_table_repo.storage_manager import StorageManager
    storage_manager = StorageManager()
    return storage_manager


@pytest.fixture
def nomenclature_manager():
    """
    Фікстура для ініціалізації NomenclatureManager
    """
    from k2.components.k2tests.k2tests.components_test_func.k2base_table_repo.nomenclature_manager \
        import NomenclatureManager
    nomenclature_manager = NomenclatureManager()
    return nomenclature_manager


@pytest.fixture
def document_manager():
    """
    Фікстура для ініціалізації DocumentManager
    """
    from k2.components.k2tests.k2tests.components_test_func.k2base_table_repo.document_manager import DocumentManager
    document_manager = DocumentManager()
    return document_manager


@pytest.fixture
def natural_person_manager():
    """
    Фікстура для ініціалізації NaturalPersonManager
    """
    from k2.components.k2tests.k2tests.components_test_func.k2base_table_repo.natural_person_manager \
        import NaturalPersonManager
    natural_person_manager = NaturalPersonManager()
    return natural_person_manager


@pytest.fixture
def patch_k2(monkeypatch):
    """
    Фікстура для мокування об'єкта K2 (як екземпляра або глобального)
    та current_user у вказаному модулі.
    Повертає мок K2 екземпляра.
    """

    def _patch(module_path, *, timezone_str='Europe/Kiev', settings_data=None, is_current_user=True):
        # Створюємо MagicMock для об'єкта K2
        mock_k2_instance = MagicMock()
        # Налаштовуємо атрибути/методи, які викликаються на об'єкті K2 в коді
        mock_k2_instance.logging_message = MagicMock()
        mock_k2_instance.log_error = "error"
        mock_k2_instance.connected_clients = {}
        mock_k2_instance.domain = "test_domain"
        mock_k2_instance.timezone = timezone(timezone_str)

        mock_k2_instance.settings = MagicMock()
        default_settings = {
            'k2production': {
                'k2task': {
                    'executionErrorPercentage': '5.0'
                }
            }
        }
        final_settings = settings_data if settings_data else default_settings

        def mock_get_settings(object_name, user_id=None):
            if object_name == 'k2production':
                return final_settings.get('k2production')
            return None

        mock_k2_instance.settings.get_settings = mock_get_settings

        monkeypatch.setattr(f"{module_path}.K2", mock_k2_instance)

        # current_user patch залишається без змін
        if is_current_user:
            mock_user = MagicMock()
            mock_user.get_id.return_value = 'test_user'
            monkeypatch.setattr(f"{module_path}.current_user", mock_user)

        # Повертаємо мок K2 екземпляра
        return mock_k2_instance

    return _patch


@pytest.fixture
def mock_request_json(monkeypatch):
    """Фікстура для мокування request.json у модулі, що тестується."""

    def _mock_request_json(module_path):
        mock_req = MagicMock()
        # Мокуємо об'єкт request у цільовому модулі
        monkeypatch.setattr(f"{module_path}.request", mock_req)
        return mock_req

    return _mock_request_json


@pytest.fixture
def mock_jsonify(monkeypatch):
    """Фікстура для мокування jsonify у модулі, що тестується."""

    def _mock_jsonify(module_path):
        mock_func = MagicMock()

        def jsonify_side_effect(data, status=200):
            if status != 200:
                return data, status
            return data

        mock_func.side_effect = jsonify_side_effect
        monkeypatch.setattr(f"{module_path}.jsonify", mock_func)
        return mock_func

    return _mock_jsonify


@pytest.fixture
def mock_socketio(monkeypatch):
    """Фікстура для мокування socketio у модулі, що тестується."""

    def _mock_socketio(module_path):
        mock = MagicMock()
        monkeypatch.setattr(f"{module_path}.socketio", mock)
        return mock

    return _mock_socketio


@pytest.fixture
def mock_logging(monkeypatch):
    """Фікстура для мокування logging у модулі, що тестується."""

    def _mock_logging(module_path):
        mock = MagicMock()
        monkeypatch.setattr(f"{module_path}.logging", mock)
        return mock

    return _mock_logging


@pytest.fixture(autouse=True)
def cleanup_after_test(db_connection, request):
    """
    Автоматична очистка таблиць після кожного тесту.
    Очікує, що в класі тесту буде атрибут TABLES — список назв таблиць.
    """
    yield

    cls = getattr(request, 'cls', None)
    if cls and hasattr(cls, 'TABLES'):
        tables = getattr(cls, 'TABLES')
        for table in tables:
            try:
                db_connection.execute(text(f"DELETE FROM {table}"))
            except Exception as ex:
                print(f"Error cleaning table {table}: {ex}")
        db_connection.commit()


# @pytest.fixture(scope="session")
# def data_migration_handbook(db_connection):
#     from k2.app import app
#     with app.test_client() as client:
#         print('🚀 Виконую міграції даних для компонент...')
#         response = client.get("/k2site/data-migrations", follow_redirects=True)
#         assert response.status_code == 200, "❌ Помилка при виконанні data-migrations"
#         yield client
